/* sane - Scanner Access Now Easy.

   pie-usb.c

   Copyright (C) 2011 Michael Rickmann <mrickma@gwdg.de>
   
   This file is part of the SANE package.
   
   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License as
   published by the Free Software Foundation; either version 2 of the
   License, or (at your option) any later version.
   
   This program is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.
   
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place - Suite 330, Boston,
   MA 02111-1307, USA.
   
   As a special exception, the authors of SANE give permission for
   additional uses of the libraries contained in this release of SANE.
   
   The exception is that, if you link a SANE library with other files
   to produce an executable, this does not by itself cause the
   resulting executable to be covered by the GNU General Public
   License.  Your use of that executable is in no way restricted on
   account of linking the SANE library code into it.
   
   This exception does not, however, invalidate any other reasons why
   the executable file might be covered by the GNU General Public
   License.
   
   If you submit changes to SANE to the maintainers to be included in
   a subsequent release, you agree by submitting the changes that
   those changes may be distributed with this exception intact.
   
   If you write modifications of your own for SANE, it is your choice
   whether to permit this exception to apply to your modifications.
   If you do not wish that, delete this exception notice. 
*/


/* list of USB descriptors */

static PIE_USB_Device_Entry pie_usb_device_list[] = {
  {0x05e3, 0x0145, 1},		/* Reflecta ProScan 7200        */
  /* Reflecta CrystalScan 7200    */
  {0, 0, 0}
};

/* values for some stereotype USB control write sequences */

static PIE_USB_Value_Data PIE_USB_Init_Sequence_1[] = {
  {VALUE_INIT_1, 0x04},
  {VALUE_INIT_2, 0xff}, {VALUE_INIT_2, 0xaa}, {VALUE_INIT_2, 0x55},
  {VALUE_INIT_2, 0x00},
  {VALUE_INIT_2, 0xff}, {VALUE_INIT_2, 0x87}, {VALUE_INIT_2, 0x78},
  {VALUE_INIT_2, 0x30},
  {VALUE_INIT_1, 0x05}, {VALUE_INIT_1, 0x04},
  {VALUE_INIT_2, 0xff},
  {0x0, 0x0}
};

static PIE_USB_Value_Data PIE_USB_Init_Sequence_2[] = {
  {VALUE_INIT_2, 0xff}, {VALUE_INIT_2, 0xaa}, {VALUE_INIT_2, 0x55},
  {VALUE_INIT_2, 0x00},
  {VALUE_INIT_2, 0xff}, {VALUE_INIT_2, 0x87}, {VALUE_INIT_2, 0x78},
  {VALUE_INIT_2, 0x00},
  {VALUE_INIT_1, 0x05}, {VALUE_INIT_1, 0x04},
  {VALUE_INIT_2, 0xff},
  {0x0, 0x0}
};

static PIE_USB_Value_Data PIE_USB_Setup_SCSI_Sequence[] = {
  {VALUE_INIT_2, 0xff}, {VALUE_INIT_2, 0xaa}, {VALUE_INIT_2, 0x55},
  {VALUE_INIT_2, 0x00},
  {VALUE_INIT_2, 0xff}, {VALUE_INIT_2, 0x87}, {VALUE_INIT_2, 0x78},
  {VALUE_INIT_2, 0xe0},
  {VALUE_INIT_1, 0x05}, {VALUE_INIT_1, 0x04},
  {VALUE_INIT_2, 0xff},
  {0x0, 0x0}
};


/* forward / prototype declarations */

static SANE_Status attach_one (const char *name);
SANE_Status
pie_usb_scsi_wrapper (int fd, const void *src, size_t src_size,
		      void *dst, size_t * dst_size);
static void pie_dump_buffer (int level, unsigned char *buf, int n);

SANE_Status (*pie_scsi_cmd) (int fd, const void *src, size_t src_size,
			     void *dst, size_t * dst_size);

static SANE_Status
sense_handler (int scsi_fd, unsigned char *result, void *arg);


/* ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~ helpers ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~ */

static SANE_Status
pie_usb_poke_bytes (char *filename, unsigned char *nums, int *len)
{
  FILE *afile;
  char buffer[1024];
  char *fres, *nptr;
  char *eptr = NULL;
  long int lnum;
  int i;

  afile = fopen (filename, "r");
  if (afile == NULL)
    {
      DBG (DBG_error, "pie_usb_poke_bytes: file %s does not exist\n",
	   filename);
      return SANE_STATUS_IO_ERROR;
    }

  i = 0;
  do
    {
      fres = fgets (buffer, sizeof (buffer), afile);
      if (fres != NULL)
	{
	  nptr = buffer;
	  errno = 0;
	  lnum = strtol (nptr, &eptr, 16);
	  while ((errno == 0) && (nptr != eptr) && (i < *len))
	    {
	      nums[i] = lnum & 0xff;
	      i++;
	      nptr = eptr;

	      errno = 0;
	      lnum = strtol (nptr, &eptr, 16);
	    }
	}
    }
  while (fres != NULL);
  *len = i;

  fclose (afile);

  return SANE_STATUS_GOOD;
}


/* ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~ low level USB ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~ */

/* ----------------------- PIE_USB_WRITE_CONTROL_SEQUENCE ------------------------ */

static SANE_Status
pie_usb_write_control_sequence (SANE_Int dn, PIE_USB_Value_Data sequ[])
{
  SANE_Int status = SANE_STATUS_GOOD;
  int i;

  DBG (DBG_info, "pie_usb_write_control_sequence writing\n");

  for (i = 0; (status == SANE_STATUS_GOOD) && (sequ[i].bValue != 0); i++)
    {
      status =
	sanei_usb_control_msg (dn, REQUEST_TYPE_OUT, REQUEST_REGISTER,
			       sequ[i].bValue, INDEX, 1, &(sequ[i].bData));
      if (status != SANE_STATUS_GOOD)
	{
	  DBG (DBG_error, "pie_usb_write_control_sequence failed\n");
	  return status;
	}
    }
  return SANE_STATUS_GOOD;
}

/* ---------------------------- PIE_USB_READY_STATE ------------------------------ */

/* Actually, there are two answers, a simple one where 1 byte with 0 or 1 is ok,   */
/* and a more elaborate one in two bytes starting with 3. The second byte signals  */
/* some condition, 0 is ready.                                                     */

static SANE_Status
pie_usb_ready_state (SANE_Int dn)
{
  SANE_Status status;
  SANE_Byte val;

  status =
    sanei_usb_control_msg (dn, REQUEST_TYPE_IN, REQUEST_REGISTER,
			   VALUE_READ_REGISTER, INDEX, 1, &val);
  if (status != SANE_STATUS_GOOD)
    {
      DBG (DBG_error, "pie_usb_ready_state failed at 1st read\n");
      return status;
    }
  DBG (DBG_info, "pie_usb_ready_state got 0x%02x at 1st read\n", val);

  if (val <= 1)
    return SANE_STATUS_GOOD;
  if (val != 3)
    {
      DBG (DBG_error, "pie_usb_ready_state failed\n");
      return SANE_STATUS_INVAL;
    }

  status =
    sanei_usb_control_msg (dn, REQUEST_TYPE_IN, REQUEST_REGISTER,
			   VALUE_READ_REGISTER, INDEX, 1, &val);
  if (status != SANE_STATUS_GOOD)
    {
      DBG (DBG_error, "pie_usb_ready_state failed at 2nd read\n");
      return status;
    }
  DBG (DBG_info, "pie_usb_ready_state got 0x%02x at 2nd read\n", val);

  if (val == 0)
    return SANE_STATUS_GOOD;
  else if (val == 8)
    return SANE_STATUS_DEVICE_BUSY;
  else if (val == 2)
    return SANE_STATUS_IO_ERROR;
  else
    return SANE_STATUS_INVAL;
}


/* --------------------------- PIE_USB_WRITE_SCSI_CMD ---------------------------- */

static SANE_Status
pie_usb_write_scsi_cmd (SANE_Int dn, const SANE_Byte cmnd[6])
{
  SANE_Status status;
  SANE_Byte mnd;
  int i;

  DBG (DBG_proc, "pie_usb_write_scsi_cmd writing 6 bytes\n");

  for (i = 0; i < 6; i++)
    {
      mnd = cmnd[i];
      status =
	sanei_usb_control_msg (dn, REQUEST_TYPE_OUT, REQUEST_REGISTER,
			       VALUE_WRITE_REGISTER, INDEX, 1, &mnd);
      if (status != SANE_STATUS_GOOD)
	{
	  DBG (DBG_error, "pie_usb_write_scsi_cmd failed at byte %d\n", i);
	  return status;
	}
    }
  return SANE_STATUS_GOOD;
}


/* ----------------------------- PIE_USB_BULK_READ ------------------------------- */

static SANE_Status
pie_usb_bulk_read (SANE_Int dn, SANE_Byte * data, size_t len)
{
  SANE_Status status;
  size_t size;
  SANE_Byte outdata[8];

  DBG (DBG_proc, "pie_usb_bulk_read requesting %lu bytes\n", (u_long) len);

  if (len == 0)

    return SANE_STATUS_GOOD;

  memset (outdata, '\0', sizeof (outdata));

  while (len)
    {
      if (len > BULKIN_MAXSIZE)
	size = BULKIN_MAXSIZE;
      else
	size = len;

      outdata[4] = (size & 0xff);
      outdata[5] = ((size >> 8) & 0xff);
      outdata[6] = ((size >> 16) & 0xff);
      outdata[7] = ((size >> 24) & 0xff);

      status =
	sanei_usb_control_msg (dn, REQUEST_TYPE_OUT, REQUEST_BUFFER,
			       VALUE_BUFFER, INDEX, sizeof (outdata),
			       outdata);
      if (status != SANE_STATUS_GOOD)
	{
	  DBG (DBG_error,
	       "pie_usb_bulk_read failed while writing command: %s\n",
	       sane_strstatus (status));
	  return status;
	}

      DBG (DBG_info,
	   "pie_usb_bulk_read trying to read %lu bytes of data\n",
	   (u_long) size);
      status = sanei_usb_read_bulk (dn, data, &size);
      if (status != SANE_STATUS_GOOD)
	{
	  DBG (DBG_error,
	       "pie_usb_bulk_read failed while reading bulk data: %s\n",
	       sane_strstatus (status));
	  return status;
	}

      DBG (DBG_info,
	   "pie_usb_bulk_read read %lu bytes, %lu remaining\n",
	   (u_long) size, (u_long) (len - size));
      len -= size;
      data += size;
    }

  DBG (DBG_info, "pie_usb_bulk_read completed\n");
  return SANE_STATUS_GOOD;
}


/* ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~ SCSI over USB ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~ */

/* ------------------------------- PIE_USB_READ ---------------------------------- */

/* note: the transfer length in a real 6-byte READ command can have two meanings:  */
/* Normally it means length in bytes, however, when an image is read it amounts to */
/* (lines * color-planes)                                                          */

static SANE_Status
pie_usb_read (int dn, const SANE_Byte * cmnd, void *buf, size_t * buf_len)
{
  SANE_Status status;
  size_t length = *buf_len;

  DBG (DBG_proc, "pie_usb_read\n");

  status = pie_usb_write_scsi_cmd (dn, cmnd);
  if (status != SANE_STATUS_GOOD)
    return status;
  status = pie_usb_ready_state (dn);
  if (status != SANE_STATUS_GOOD)
    return status;

  status = pie_usb_bulk_read (dn, buf, length);
  if (status != SANE_STATUS_GOOD)
    return status;

  return pie_usb_ready_state (dn);
}

/* ------------------------------- PIE_USB_WRITE --------------------------------- */

static SANE_Status
pie_usb_write (int dn, const SANE_Byte * cmnd, size_t length)
{
  SANE_Status status;
  SANE_Byte mnd;
  size_t i;

  DBG (DBG_info, "pie_usb_write\n");
  if (length <= 6)
    return SANE_STATUS_GOOD;

  status = pie_usb_write_scsi_cmd (dn, cmnd);
  if (status != SANE_STATUS_GOOD)
    return status;
  status = pie_usb_ready_state (dn);
  if (status != SANE_STATUS_GOOD)
    return status;

  DBG (DBG_info, "pie_usb_write: now writing %lu bytes\n",
       (u_long) length - 6);
  for (i = 6; i < length; i++)
    {
      mnd = cmnd[i];
      status =
	sanei_usb_control_msg (dn, REQUEST_TYPE_OUT, REQUEST_REGISTER,
			       VALUE_WRITE_REGISTER, INDEX, 1, &mnd);
      if (status != SANE_STATUS_GOOD)
	{
	  DBG (DBG_error, "pie_usb_write failed at byte %lu\n",
	       (unsigned long) length);
	  return status;
	}
    }

  return pie_usb_ready_state (dn);
}


/* ------------------------------ PIE_USB_COMMAND -------------------------------- */

static SANE_Status
pie_usb_command (int dn, const SANE_Byte * cmnd)
{
  SANE_Status status;

  DBG (DBG_info, "pie_usb_command\n");

  status = pie_usb_write_scsi_cmd (dn, cmnd);
  if (status != SANE_STATUS_GOOD)
    return status;

  return pie_usb_ready_state (dn);
}

/* ---------------------------- PIE_USB_SCSI_WRAPPER ----------------------------- */

/* Entry for SCSI over USB command wrapping                                        */

SANE_Status
pie_usb_scsi_wrapper (int fd, const void *src, size_t src_size, void *dst,
		      size_t * dst_size)
{
  SANE_Status status = SANE_STATUS_GOOD;
  const SANE_Byte *cmnd = src;

  if (cmnd[0] == INQUIRY)
    {
      status = pie_usb_write_control_sequence (fd, PIE_USB_Init_Sequence_1);
      if (status != SANE_STATUS_GOOD)
	return status;
      status = pie_usb_write_control_sequence (fd, PIE_USB_Init_Sequence_2);
      if (status != SANE_STATUS_GOOD)
	return status;
    }
  status = pie_usb_write_control_sequence (fd, PIE_USB_Setup_SCSI_Sequence);
  if (status != SANE_STATUS_GOOD)
    return status;

  switch (cmnd[0])
    {
    case TEST_UNIT_READY:
      DBG (DBG_info, "pie_usb_scsi_wrapper doing TEST_UNIT_READY\n");
      status = pie_usb_command (fd, cmnd);
      break;
    case REQUEST_SENSE:
      DBG (DBG_info, "pie_usb_scsi_wrapper doing REQUEST_SENSE\n");
      status = pie_usb_read (fd, cmnd, dst, dst_size);
      break;
    case READ:
      DBG (DBG_info, "pie_usb_scsi_wrapper doing READ\n");
      status = pie_usb_read (fd, cmnd, dst, dst_size);
      break;
    case WRITE:
      DBG (DBG_info, "pie_usb_scsi_wrapper doing WRITE\n");
      status = pie_usb_write (fd, cmnd, src_size);
      break;
    case INQUIRY:
      DBG (DBG_info, "pie_usb_scsi_wrapper doing INQUIRY\n");
      status = pie_usb_read (fd, cmnd, dst, dst_size);
      break;
    case PARAM:
      DBG (DBG_info, "pie_usb_scsi_wrapper doing PARAM\n");
      status = pie_usb_read (fd, cmnd, dst, dst_size);
      break;
    case MODE:
      DBG (DBG_info, "pie_usb_scsi_wrapper doing MODE\n");
      status = pie_usb_write (fd, cmnd, src_size);
      break;
    case RESERVE_UNIT:
      DBG (DBG_info, "pie_usb_scsi_wrapper doing RESERVE_UNIT\n");
      status = pie_usb_command (fd, cmnd);
      break;
    case RELEASE_UNIT:
      DBG (DBG_info, "pie_usb_scsi_wrapper doing RELEASE_UNIT\n");
      status = pie_usb_command (fd, cmnd);
      break;
    case PIE_COPY:
      DBG (DBG_info, "pie_usb_scsi_wrapper doing PIE_COPY\n");
      status = pie_usb_read (fd, cmnd, dst, dst_size);
      break;
    case SCAN:
      DBG (DBG_info, "pie_usb_scsi_wrapper doing SCAN\n");
      status = pie_usb_command (fd, cmnd);
      break;
    case PIE_IMAGE_CLOSE:
      DBG (DBG_info, "pie_usb_scsi_wrapper doing PIE_IMAGE_CLOSE\n");
      status = pie_usb_write (fd, cmnd, src_size);
      break;
    case PIE_READ_CALIBRATION:
      DBG (DBG_info, "pie_usb_scsi_wrapper doing PIE_READ_CALIBRATION\n");
      status = pie_usb_read (fd, cmnd, dst, dst_size);
      break;
    case PIE_WRITE_CALIBRATION:
      DBG (DBG_info, "pie_usb_scsi_wrapper doing PIE_WRITE_CALIBRATION\n");
      status = pie_usb_write (fd, cmnd, src_size);
      break;
    case PIE_READ_STATUS:
      DBG (DBG_info, "pie_usb_scsi_wrapper doing PIE_READ_STATUS\n");
      status = pie_usb_read (fd, cmnd, dst, dst_size);
      break;
    default:
      DBG (DBG_info, "pie_usb_scsi_wrapper failed for command 0x%02x\n",
	   cmnd[0]);
      status = SANE_STATUS_INVAL;
      break;
    }

  return status;
}


/* ---------------------------- PIE_USB_REQUEST_SENSE ---------------------------- */

static SANE_Status
pie_usb_request_sense (int dn, uint32_t * kascq)
{
  unsigned char buffer[16];	/* for PIE's sense */
  size_t size;
  SANE_Status status;

  DBG (DBG_proc, "pie_usb_request_sense\n");

  size = 14;
  set_RS_allocation_length (request_senseC, size);

  status =
    pie_usb_scsi_wrapper (dn, request_senseC, sizeof (request_senseC), buffer,
			  &size);
  if (status != SANE_STATUS_GOOD)
    {
      DBG (DBG_error, "pie_usb_request_sense failed\n");
      return status;
    }
  *kascq = ((int) get_RS_sense_key (buffer) << 16) +
    ((int) get_RS_ASC (buffer) << 8) + (int) get_RS_ASCQ (buffer);
  sense_handler (dn, buffer, NULL);

  return SANE_STATUS_GOOD;
}


/* -------------------------- PIE_USB_CLOSE_IMAGE_READ --------------------------- */

/* PIE vendor specific, called after image aquisition, waits for ready */

static SANE_Status
pie_usb_close_image_read (Pie_Scanner * scanner)
{
  unsigned char buffer[16];
  unsigned char *data;
  SANE_Status status;
  size_t size = image_closeC[4];
  int cnt = 0;

  DBG (DBG_proc, "pie_usb_close_image_read\n");

  memcpy (buffer, image_closeC, sizeof (image_closeC));

  data = buffer + sizeof (image_closeC);
  memset (data, 0, size);
  data[0] = 2;
  size += sizeof (image_closeC);

  do
    {
      status = pie_usb_scsi_wrapper (scanner->sfd, buffer, size, NULL, NULL);

      if (status == SANE_STATUS_GOOD)
	return status;

      if (cnt == 1)
	DBG (DBG_info2,
	     "pie_usb_close_image_read: scanner reports %s, waiting ...\n",
	     sane_strstatus (status));

      usleep (TUR_WAIT_TIME);
      cnt++;
    }
  while (cnt < 20);		/* maximally 20 * 0.5 = 10 sec */

  return status;
}

/* ---------------------------- PIE_USB_DUMP_COPY ----------------------------- */

static SANE_Status
pie_usb_dump_copy (Pie_Scanner * scanner)
{
  unsigned char buffer[5340];	/* TODO is this harcoded ? */
  size_t size_read = 5340;
  int cnt = 0;
  SANE_Status status;

  DBG (DBG_proc, "pie_usb_dump_copy\n");

  do
    {
      status =
	pie_usb_scsi_wrapper (scanner->sfd, pie_copyC, sizeof (pie_copyC),
			      buffer, &size_read);
      if (status == SANE_STATUS_GOOD)
	return status;

      if (cnt == 1)
	DBG (DBG_info2,
	     "pie_usb_close_image_read: scanner reports %s, waiting ...\n",
	     sane_strstatus (status));

      usleep (TUR_WAIT_TIME);
      cnt++;
    }
  while (cnt < 10);		/* maximally 10 * 0.5 = 5 sec */

  return status;
}

/* --------------------------- PIE_USB_READ_STATUS ---------------------------- */

/* PIE vendor specific 0xdd, called during idle, 
   initial internal calibration, ...,  reads e.g. button state */

static SANE_Status
pie_usb_read_status (int dn, unsigned char *buf)
{
  SANE_Status status;
  size_t size;

  DBG (DBG_proc, "pie_usb_read_status\n");

  size = 11;

  status =
    pie_usb_scsi_wrapper (dn, read_statusC, sizeof (read_statusC), buf,
			  &size);
  return status;
}


/* ---------------------------- PIE_USB_WAIT_SCANNER ----------------------------- */

static SANE_Status
pie_usb_wait_scanner (Pie_Scanner * scanner, int secs)
{
  SANE_Status status;
  int cnt = secs * 16;

  DBG (DBG_proc, "pie_usb_wait_scanner\n");

  do
    {
      status = pie_usb_scsi_wrapper (scanner->sfd, test_unit_ready.cmd,
				     test_unit_ready.size, NULL, NULL);
      if (status == SANE_STATUS_GOOD)
	return status;
      if (cnt == 0)
	{
	  DBG (DBG_warning, "pie_usb_wait_scanner timed out\n");
	  return status;
	}
      usleep (62500);
      cnt--;

    }
  while (status == SANE_STATUS_DEVICE_BUSY);

  DBG (DBG_error,
       "pie_usb_wait_scanner failed: %s\n", sane_strstatus (status));
  return status;
}


/* ------------------------ PIE_USB_SEND_CALIBRATION -------------------------- */

/* TODO currently faked calibration data */

static unsigned char pie_usb_normal_expos[] =
  { 0x79, 0x0e, 0xff, 0x0b, 0x6c, 0x0c };

static unsigned char pie_usb_normal_gainsp[] =
  { 0x3f, 0x3d, 0x39, 0x07, 0x00, 0x00 };

static unsigned char pie_usb_hiqual_expos[] =
  { 0x89, 0x2c, 0xb7, 0x1e, 0xca, 0x17 };

static unsigned char pie_usb_hiqual_gainsp[] =
  { 0x25, 0x25, 0x28, 0x07, 0x00, 0x00 };

/* This routine contains the vendor SCSI commands 0xd7 for reading and 0xdc
   for writing. As in the logs we first read 103 bytes and then write 23 bytes.
   It seems that we should take the last 5 bytes of the read data also as the
   last to be written. So only 18 bytes of faked calibration data is pasted.    */


static SANE_Status
pie_usb_send_calibration (Pie_Scanner * scanner)
{
  PIE_USB_Calibration_Read in_buf;
  PIE_USB_Calibration_Send out_buf;
  size_t size_read = 103;
  size_t size_write = 23;
  SANE_Status status;
  unsigned char pokebuf[64];
  int pokesiz;

  DBG (DBG_proc, "pie_usb_send_calibration\n");

  status =			/* 103 bytes */
    pie_usb_scsi_wrapper (scanner->sfd, read_calibrationC,
			  sizeof (read_calibrationC), &in_buf, &size_read);
  if (status != SANE_STATUS_GOOD)
    return status;

  DBG (DBG_info, "pie_usb_send_calibration received:\n");
  DBG_DUMP (DBG_info, (unsigned char *) &in_buf, 103);

  memcpy (&(out_buf.scsi_cmd), write_calibrationC, 6);
  size_write += 6;

  memcpy (&(out_buf.copy_1), &(in_buf.copy_1), sizeof (out_buf.copy_1));
  memcpy (&(out_buf.copy_2), &(in_buf.copy_2), sizeof (out_buf.copy_2));

  if ((scanner->cal_mode & CAL_MODE_FILM_HIQUAL) == 0)
    {
      memcpy (&(out_buf.red_texp), pie_usb_normal_expos,
	      sizeof (pie_usb_normal_expos));
      memcpy (&(out_buf.red_gain), pie_usb_normal_gainsp,
	      sizeof (pie_usb_normal_gainsp));
    }
  else
    {
      memcpy (&(out_buf.red_texp), pie_usb_hiqual_expos,
	      sizeof (pie_usb_normal_expos));
      memcpy (&(out_buf.red_gain), pie_usb_hiqual_gainsp,
	      sizeof (pie_usb_normal_gainsp));
    }

  pokesiz = 12;
  status = pie_usb_poke_bytes ("/tmp/calbytes.txt", pokebuf, &pokesiz);
  if (status == SANE_STATUS_GOOD)
    {
      memcpy (&(out_buf.red_texp), &pokebuf[0], 6);
      memcpy (&(out_buf.red_gain), &pokebuf[6], 6);
    }

  DBG (DBG_info, "pie_usb_send_calibration sending:\n");
  DBG_DUMP (DBG_info, (unsigned char *) &out_buf, size_write);

  status =
    pie_usb_scsi_wrapper (scanner->sfd, &out_buf, size_write, NULL, NULL);

  return status;
}

/* ----------------------------- PIE_USB_ATTACH_OPEN ----------------------------- */

/* called during initialization, tries to open and identify an USB scanner	   */

static SANE_Status
pie_usb_attach_open (SANE_String_Const devname, SANE_Int * dn,
		     SANE_Int * model)
{
  SANE_Status status;
  SANE_Int vendor, product;
  int i;

  DBG (DBG_proc, "pie_usb_attach_open: opening `%s'\n", devname);
  status = sanei_usb_open (devname, dn);
  if (status != SANE_STATUS_GOOD)
    {
      DBG (DBG_error, "pie_usb_attach_open: sanei_usb_open failed\n");
      return status;
    }
  DBG (DBG_info, "pie_usb_attach_open: USB device `%s' successfully opened\n",
       devname);

  status = sanei_usb_get_vendor_product (*dn, &vendor, &product);
  if (status != SANE_STATUS_GOOD)
    {
      DBG (DBG_error,
	   "pie_usb_attach_open: couldn't get vendor and product ids of device `%s': %s\n",
	   devname, sane_strstatus (status));
      return status;
    }

  *model = 0;
  for (i = 0; pie_usb_device_list[i].model != 0; i++)
    {
      if (vendor == pie_usb_device_list[i].vendor &&
	  product == pie_usb_device_list[i].product)
	{
	  *model = pie_usb_device_list[i].model;
	  break;
	}
    }
  if (model == 0)
    {
      DBG (DBG_error,
	   "pie_usb_attach_open: vendor 0x%04x product 0x%04x is not supported by this backend\n",
	   vendor, product);
      status = SANE_STATUS_INVAL;
    }

  return status;
}



/* -------------------------------- TRY_ATTACH_USB ------------------------------- */

/* callback function for sanei_config_attach_matching_devices which has checked    */
/* for SCSI scanners. Here we do it for USB ones                                   */

static SANE_Status
try_attach_usb (const char *name)
{
  sanei_usb_attach_matching_devices (name, attach_one);
  return SANE_STATUS_GOOD;
}


/* ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~ high level functions ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~ */

/* ------------------------ PIE_USB_IS_SCANNER_READY -------------------------- */

static SANE_Status
pie_usb_is_scanner_ready (int dn)
{
  unsigned char buffer[16];
  SANE_Status status, state;
  uint32_t sense_kascq;
  int wait_cnt = 240;		/* 240 * 0.5 = 120 seconds */

  DBG (DBG_proc, "pie_usb_is_scanner_ready\n");

  do
    {
      status = pie_usb_scsi_wrapper (dn, test_unit_ready.cmd,
				     test_unit_ready.size, NULL, NULL);
      if (status == SANE_STATUS_IO_ERROR)	/* Not Ready - Warming Up ? */
	{
	  state = pie_usb_request_sense (dn, &sense_kascq);
	  if (state != SANE_STATUS_GOOD)
	    return state;
	  if (sense_kascq != 0x020401)
	    return status;
	  else
	    status = SANE_STATUS_DEVICE_BUSY;
	}

      if (status == SANE_STATUS_DEVICE_BUSY)
	{
	  usleep (TUR_WAIT_TIME);
	  wait_cnt--;
	}

      memset (buffer, 0, 11);

      status = pie_usb_read_status (dn, buffer);
      if (status == SANE_STATUS_IO_ERROR)	/* Not Ready - Warming Up ? */
	{
	  state = pie_usb_request_sense (dn, &sense_kascq);
	  if (state != SANE_STATUS_GOOD)
	    return state;
	  if (sense_kascq != 0x020401)
	    return status;
	  else
	    status = SANE_STATUS_DEVICE_BUSY;
	}

      DBG_DUMP (DBG_info, buffer, 11);

      if ((buffer[9] != 0) || (buffer[10] != 0))
	status = SANE_STATUS_DEVICE_BUSY;

      if (status == SANE_STATUS_DEVICE_BUSY)
	{
	  usleep (TUR_WAIT_TIME);
	  wait_cnt--;
	}
    }
  while ((status == SANE_STATUS_DEVICE_BUSY) && (wait_cnt > 0));

  return status;
}


/* ------------------------------ PIE_USB_SCAN -------------------------------- */

static SANE_Status
pie_usb_scan (Pie_Scanner * scanner, int start)
{
  SANE_Status status, state;
  uint32_t sense_kascq;

  DBG (DBG_proc, "pie_usb_scan: %d\n", start);

  set_scan_cmd (scan.cmd, start);
  if (start)
    {
      /* wait upto X seconds until returned to start position */
      status = pie_usb_wait_scanner (scanner, 15);
      if (status != SANE_STATUS_GOOD)
	return status;

      do
	{
	  status = pie_usb_scsi_wrapper (scanner->sfd, scan.cmd, scan.size,
					 NULL, NULL);
	  if (status)
	    {
	      DBG (DBG_error, "pie_usb_scan: received %s\n",
		   sane_strstatus (status));
	      if (status == SANE_STATUS_IO_ERROR)
		{
		  state = pie_usb_request_sense (scanner->sfd, &sense_kascq);
		  if (state != SANE_STATUS_GOOD)
		    return state;
		  if (sense_kascq == 0x020401)	/* Not Ready - Warming Up */
		    status = SANE_STATUS_DEVICE_BUSY;
		  else if (sense_kascq == 0x068200)	/* calibration disable not granted */
		    status = SANE_STATUS_GOOD;
		}
	      if (status == SANE_STATUS_DEVICE_BUSY)
		usleep (SCAN_WARMUP_WAIT_TIME);
	    }
	}
      while (status == SANE_STATUS_DEVICE_BUSY);

      return status;
    }
  else
    {
      status = pie_usb_scsi_wrapper (scanner->sfd, scan.cmd, scan.size,
				     NULL, NULL);
      if (status == SANE_STATUS_IO_ERROR)
	{
	  state = pie_usb_request_sense (scanner->sfd, &sense_kascq);
	  if (state != SANE_STATUS_GOOD)
	    return state;
	  if (sense_kascq != 0x0b0006)	/* ABORT message from initiator */
	    return status;
	}
      return SANE_STATUS_GOOD;
    }
}


/* ---------------------------- PIE_USB_CALIBRATE ----------------------------- */

/* TODO this is mostly based on guesses */
/* Guessed sequence of events:
   1) 1 line of image data is read
   2) test unit ready
   4)  13 lines are read
   5)  something should be learned from that
   6)  calibration info is sent to the scanner
   7) perhaps repeat steps 4) to 6)
   8) read remaining lines of totally 45
   scanner->device->cal_info[any].num_lines contains 45
*/

#define SKIP	150		/* pixels at each end of line */
#define FULL	0xf000		/* full calibration */

static SANE_Status
pie_usb_calibrate (Pie_Scanner * scanner)
{
  unsigned int red_result = 0;
  unsigned int green_result = 0;
  unsigned int blue_result = 0;
  unsigned int ired_result = 0;
  unsigned int *result = NULL;
  uint64_t sq_red_result = 0;
  uint64_t sq_green_result = 0;
  uint64_t sq_blue_result = 0;
  uint64_t sq_ired_result = 0;
  uint64_t *sq_result = NULL;
  uint64_t val;
  int oval, oval1, oval2;
  double dval;
  unsigned char *rcv_buffer, *rcv_ptr;
  int rcv_length, rcv_lines, rcv_bits;
  int pixels_per_line, average_lines;
  size_t size;
  SANE_Status status;
  int i, j, k, n;

  DBG (DBG_proc, "pie_usb_calibrate\n");

  pixels_per_line = scanner->device->cal_info[0].pixels_per_line;
  rcv_length = pixels_per_line;
  average_lines = 0xffff / (pixels_per_line - 2 * SKIP);

  rcv_bits = scanner->device->cal_info[0].receive_bits;
  if (rcv_bits > 8)
    rcv_length *= 2;		/* 2 bytes / sample */

  rcv_lines = scanner->device->cal_info[0].num_lines;

  if (scanner->colormode == RGB)
    {
      rcv_lines *= 4;
      rcv_length += 2;		/* 2 bytes for index at front of data (only in RGB??) */
    }
  else
    return SANE_STATUS_INVAL;

  /* alllocate buffers for the receive data, the result buffers, and for the send data */
  rcv_buffer = (unsigned char *) malloc (rcv_length);

  if (!rcv_buffer)
    {
      /* at least one malloc failed, so free all buffers (free accepts NULL) */
      status = SANE_STATUS_NO_MEM;
      goto freed_end;
    }

  status = pie_wait_scanner (scanner);
  if (status != SANE_STATUS_GOOD)
    goto freed_end;

  set_read_length (sread.cmd, 1);
  size = rcv_length;

  DBG (DBG_info, "pie_usb_calibrate: trying a line (%lu bytes)\n",
       (u_long) size);

  for (i = 0; i < 4; i++)
    {
      status =
	pie_usb_scsi_wrapper (scanner->sfd, sread.cmd, sread.size,
			      rcv_buffer, &size);
      if (status != SANE_STATUS_GOOD)
	goto freed_end;
    }
  rcv_lines -= 4;

  status = pie_wait_scanner (scanner);
  if (status != SANE_STATUS_GOOD)
    return status;

  for (k = 0; k < average_lines; k++)
    {
      DBG (DBG_info, "pie_usb_calibrate: reading 1 line (%lu bytes)\n",
	   (u_long) size);
      for (i = 0; i < 4; i++)
	{
	  status =
	    pie_usb_scsi_wrapper (scanner->sfd, sread.cmd, sread.size,
				  rcv_buffer, &size);
	  if (status != SANE_STATUS_GOOD)
	    goto freed_end;

	  if (*rcv_buffer == 'R')
	    {
	      result = &red_result;
	      sq_result = &sq_red_result;
	    }
	  else if (*rcv_buffer == 'G')
	    {
	      result = &green_result;
	      sq_result = &sq_green_result;
	    }
	  else if (*rcv_buffer == 'B')
	    {
	      result = &blue_result;
	      sq_result = &sq_blue_result;
	    }
	  else if (*rcv_buffer == 'I')
	    {
	      result = &ired_result;
	      sq_result = &sq_ired_result;
	    }
	  else
	    {
	      DBG (DBG_error, "pie_perform_cal: invalid index byte (%02x)\n",
		   *rcv_buffer);
	      DBG_DUMP (DBG_error, rcv_buffer, 32);
	      goto freed_end;
	    }
	  rcv_ptr = rcv_buffer + 2 + SKIP;
	  for (j = SKIP; j < pixels_per_line - SKIP; j++)
	    {
	      val = *rcv_ptr++;
	      if (rcv_bits > 8)
		val += (*rcv_ptr++) << 8;
	      *result += val;
	      *sq_result += val * val;
	    }
	}
      rcv_lines -= 4;
    }

  n = (pixels_per_line - 2 * SKIP) * average_lines;

  oval = red_result / n;
  dval =
    (double) sq_red_result -
    (double) red_result *(double) red_result / (double) n;
  if (dval > 0)
    oval1 = sqrt (dval / (double) (n - 1));
  else
    oval1 = 0;
  oval2 = oval;
  if (oval2 > FULL)
    oval2 = FULL;
  oval2 = ((FULL - oval2) * FULL) / oval2;
  DBG (DBG_info,
       "pie_usb_calibrate: red average %d = 0x%04x, SD %d = 0x%04x, suggested calibration %d = 0x%04x\n",
       oval, oval, oval1, oval1, oval2, oval2);

  oval = green_result / n;
  dval =
    (double) sq_green_result -
    (double) green_result *(double) green_result / (double) n;
  if (dval > 0)
    oval1 = sqrt (dval / (double) (n - 1));
  else
    oval1 = 0;
  oval2 = oval;
  if (oval2 > FULL)
    oval2 = FULL;
  oval2 = ((FULL - oval2) * FULL) / oval2;
  DBG (DBG_info,
       "pie_usb_calibrate: green average %d = 0x%04x, SD %d = 0x%04x, suggested calibration %d = 0x%04x\n",
       oval, oval, oval1, oval1, oval2, oval2);

  oval = blue_result / n;
  dval =
    (double) sq_blue_result -
    (double) blue_result *(double) blue_result / (double) n;
  if (dval > 0)
    oval1 = sqrt (dval / (double) (n - 1));
  else
    oval1 = 0;
  oval2 = oval;
  if (oval2 > FULL)
    oval2 = FULL;
  oval2 = ((FULL - oval2) * FULL) / oval2;
  DBG (DBG_info,
       "pie_usb_calibrate: blue average %d = 0x%04x, SD %d = 0x%04x, suggested calibration %d = 0x%04x\n",
       oval, oval, oval1, oval1, oval2, oval2);

  oval = ired_result / n;
  dval =
    (double) sq_ired_result -
    (double) ired_result *(double) ired_result / (double) n;
  if (dval > 0)
    oval1 = sqrt (dval / (double) (n - 1));
  else
    oval1 = 0;
  oval2 = oval;
  if (oval2 > FULL)
    oval2 = FULL;
  oval2 = ((FULL - oval2) * FULL) / oval2;
  DBG (DBG_info,
       "pie_usb_calibrate: infrared average %d = 0x%04x, SD %d = 0x%04x, suggested calibration %d = 0x%04x\n",
       oval, oval, oval1, oval1, oval2, oval2);


  status = pie_usb_send_calibration (scanner);
  if (status != SANE_STATUS_GOOD)
    return status;

  while (rcv_lines > 0)
    {
      status =
	pie_usb_scsi_wrapper (scanner->sfd, sread.cmd, sread.size,
			      rcv_buffer, &size);
      if (status != SANE_STATUS_GOOD)
	goto freed_end;
      rcv_lines--;
    }

freed_end:			/* ugly, but this may be one of the cases ... */
  free (rcv_buffer);
  return status;
}


/*-------------------------- PIE_USB_MODE SELECT -------------------------------*/

static SANE_Status
pie_usb_mode_select (Pie_Scanner * scanner)
{

  SANE_Status status;
  unsigned char buffer[128];
  size_t size;
  unsigned char *data;
  int i, cal;

  DBG (DBG_proc, "pie_usb_mode_select\n");

  size = 16;

  set_mode_length (smode.cmd, size);

  memcpy (buffer, smode.cmd, smode.size);

  data = buffer + smode.size;
  memset (data, 0, size);

  /* size of data */
  data[1] = size - 2;

  /* set resolution required */
  set_data (data, 2, scanner->resolution, 2);

  /* set color filter and color depth */
  switch (scanner->colormode)
    {
    case RGB:
      if (scanner->device->inquiry_filters & INQ_ONE_PASS_COLOR)
	{
	  data[4] = INQ_ONE_PASS_COLOR;
	  scanner->cal_filter = FILTER_RED | FILTER_GREEN | FILTER_BLUE;
	}
      else
	{
	  DBG (DBG_error,
	       "pie_usb_mode_select: support for multipass color not yet implemented\n");
	  return SANE_STATUS_UNSUPPORTED;
	}
      if (scanner->val[OPT_BIT_DEPTH].w == 16)
	data[5] = INQ_COLOR_DEPTH_16;
      else
	data[5] = INQ_COLOR_DEPTH_8;
      break;

    case GRAYSCALE:
      /* choose which filter to use for monochrome mode */
      if (scanner->device->inquiry_filters & INQ_FILTER_NEUTRAL)
	{
	  data[4] = FILTER_NEUTRAL;
	  scanner->cal_filter = FILTER_NEUTRAL;
	}
      else if (scanner->device->inquiry_filters & INQ_FILTER_GREEN)
	{
	  data[4] = FILTER_GREEN;
	  scanner->cal_filter = FILTER_GREEN;
	}
      else if (scanner->device->inquiry_filters & INQ_FILTER_RED)
	{
	  data[4] = FILTER_RED;
	  scanner->cal_filter = FILTER_RED;
	}
      else if (scanner->device->inquiry_filters & INQ_FILTER_BLUE)
	{
	  data[4] = FILTER_BLUE;
	  scanner->cal_filter = FILTER_BLUE;
	}
      else
	{
	  DBG (DBG_error,
	       "pie_usb_mode_select: scanner doesn't appear to support monochrome\n");
	  return SANE_STATUS_UNSUPPORTED;
	}

      data[5] = INQ_COLOR_DEPTH_8;
    }

  /* choose color packing method */
  if (scanner->device->inquiry_color_format & INQ_COLOR_FORMAT_INDEX)
    data[6] = INQ_COLOR_FORMAT_INDEX;
  else if (scanner->device->inquiry_color_format & INQ_COLOR_FORMAT_LINE)
    data[6] = INQ_COLOR_FORMAT_LINE;
  else
    {
      DBG (DBG_error,
	   "pie_usb_mode_select: support for pixel packing not yet implemented\n");
      return SANE_STATUS_UNSUPPORTED;
    }

  /* choose data format */
  if (scanner->device->inquiry_image_format & INQ_IMG_FMT_INTEL)
    data[8] = INQ_IMG_FMT_INTEL;
  else
    {
      DBG (DBG_error,
	   "pie_usb_mode_select: support for Motorola format not yet implemented\n");
      return SANE_STATUS_UNSUPPORTED;
    }

  /* set required calibration and quality */
  i = 0;
  while (scanner->device->speed_list[i] != NULL)
    {
      if (strcmp (scanner->device->speed_list[i], scanner->val[OPT_SPEED].s)
	  == 0)
	break;
      i++;
    }

  if (scanner->device->speed_list[i] == NULL)
    i = 0;			/* neither should happen */
  if (i > 2)
    i = 2;

  cal = i;
  if (cal == 1)
    {
      if (scanner->val[OPT_PREVIEW].w == SANE_TRUE)
	cal = CAL_MODE_FILM_NORMAL;
      else
	cal = CAL_MODE_FILM_HIQUAL;
    }
  /* skip calibration if no quality in this or last scan */
  if ((cal == CAL_MODE_FILM_NORMAL) &&
      ((scanner->cal_mode & CAL_MODE_FILM_HIQUAL) == CAL_MODE_FILM_NORMAL))
    cal = CAL_MODE_FILM_SKIP;

  data[9] = cal;
  scanner->cal_mode = cal;

  /* set required halftone pattern */
  i = 0;
  while (scanner->device->halftone_list[i] != NULL)
    {
      if (strcmp
	  (scanner->device->halftone_list[i],
	   scanner->val[OPT_HALFTONE_PATTERN].s) == 0)
	break;
      i++;
    }

  if (scanner->device->halftone_list[i] == NULL)
    data[12] = 0;		/* halftone pattern */
  else
    data[12] = i;

  data[13] = SANE_UNFIX (scanner->val[OPT_THRESHOLD].w) * 255 / 100;	/* lineart threshold */

  data[14] = 0x10;

  DBG (DBG_info, "pie_usb_mode_select: speed %02x\n", data[9]);
  DBG (DBG_info, "pie_usb_mode_select: halftone %d\n", data[12]);
  DBG (DBG_info, "pie_usb_mode_select: threshold %02x\n", data[13]);

  DBG (DBG_info, "pie_usb_mode_select sending:\n");
  DBG_DUMP (DBG_info, data, size);

  status =
    (*pie_scsi_cmd) (scanner->sfd, buffer, smode.size + size, NULL, NULL);
  if (status)
    {
      DBG (DBG_error,
	   "pie_usb_mode_select: write command returned status %s\n",
	   sane_strstatus (status));
    }

  return status;
}


/* --------------------- PIE_USB_READER_PROCESS_INDEXED ----------------------- */

static int
pie_usb_reader_process_indexed (Pie_Scanner * scanner, FILE * fp)
{
  char idx_char[4] = { 'R', 'G', 'B', 'I' };
  char *idx_ptr[4];
  char *iend_ptr[4];
  int idx_buf[4];
  int idx_found;
  unsigned char *buffer = NULL;
  unsigned char *reorder = NULL;
  unsigned char *wrt_ptr, *end_ptr;
  unsigned char *dest, *src[4];
  int bytes_per_line, bytes_per_color;
  int chunk_lines, lines_todo;
  int read_lines, write_lines;
  int status;
  int irgb;			/* number of color planes = values per pixel */
  int irgb_out;			/* limit output as long as SANE disregards infrared */
  int request_data, i, j;
  size_t chunk_size, size;

  DBG (DBG_read,
       "pie_usb_reader_process_indexed reading %d lines of %d bytes/line\n",
       scanner->params.lines, scanner->params.bytes_per_line);

  bytes_per_color = scanner->bytes_per_line + 2;
  switch (scanner->colormode)
    {
    case RGB:
      irgb = 3;
      irgb_out = 3;
      bytes_per_line = scanner->params.bytes_per_line + 6;
      break;
    case RGBI:
      irgb = 4;
#ifdef SANE_FRAME_RGBI
      irgb_out = 4;
      bytes_per_line = scanner->params.bytes_per_line + 8;
#else
      irgb_out = 3;
      bytes_per_line = scanner->params.bytes_per_line + bytes_per_color + 6;
#endif
      break;
    default:
      irgb = 1;
      irgb_out = 1;		/* assume filter bytes not present */
      bytes_per_line = scanner->params.bytes_per_line;
    }

  /* split the image read into reasonably sized chunks */
  chunk_lines = (scanner->params.lines + 7) / 8;
  chunk_size = chunk_lines * bytes_per_line;

  if (chunk_size > BUFFER_MAXSIZE)	/* hardware limitation ? */
    {
      chunk_lines = BUFFER_MAXSIZE / bytes_per_line;
      chunk_size = chunk_lines * bytes_per_line;
    }

  size = (scanner->bytes_per_line + 2) *	/* deskewing needs a minimum */
    (scanner->filter_offset1 + (scanner->filter_offset2 + 3) * 2);
  if (chunk_size < size)
    {
      chunk_lines = (size + bytes_per_line - 1) / bytes_per_line;
      chunk_size = chunk_lines * bytes_per_line;
    }

  if (chunk_lines > scanner->params.lines)	/* not larger than image */
    {
      chunk_lines = scanner->params.lines;
      chunk_size = chunk_lines * bytes_per_line;
    }

  buffer = malloc (chunk_size * 2);
  reorder = malloc (bytes_per_line);
  if (!buffer || !reorder)
    {
      free (buffer);
      free (reorder);
      return SANE_STATUS_NO_MEM;
    }

  /* read one buffer in advance */
  lines_todo = chunk_lines;
  set_read_length (sread.cmd, lines_todo * irgb);
  size = lines_todo * bytes_per_line;
  do
    {
      status =
	pie_usb_scsi_wrapper (scanner->sfd, sread.cmd, sread.size,
			      buffer, &size);
    }
  while (status);

  DBG_DUMP (DBG_dump, buffer, 32);

  write_lines = scanner->params.lines;
  read_lines = write_lines - lines_todo;
  wrt_ptr = buffer + chunk_size;
  end_ptr = wrt_ptr + chunk_size;
  for (i = 0; i < irgb; i++)
    {
      idx_ptr[i] = (char *) buffer;
      iend_ptr[i] = idx_ptr[i] + chunk_size;
      idx_buf[i] = 1;
      src[i] = NULL;
    }
  request_data = 0;
  idx_found = 0;

  while (write_lines)
    {
      if (irgb > 1)		/* RGB, RGBI cases */
	{
	  for (i = 0; i < irgb; i++)	/* find indices */
	    {
	      while (src[i] == NULL)
		{
		  if (*idx_ptr[i] == idx_char[i])
		    {
		      src[i] = (unsigned char *) idx_ptr[i] + 2;
		      idx_found++;
		    }
		  /* advance pointers unconditionally */
		  idx_ptr[i] += bytes_per_color;
		  if (idx_ptr[i] >= iend_ptr[i])
		    {
		      /* check for wrap */
		      if (idx_ptr[i] >= (char *) end_ptr)
			idx_ptr[i] = (char *) buffer;
		      /* maintain private "end of buffer" */
		      iend_ptr[i] = idx_ptr[i] + chunk_size;
		      idx_buf[i]--;
		      /* request buffer fill if necessary */
		      if (idx_buf[i] == 0)
			{
			  request_data = 1;
			  break;
			}
		    }
		}
	    }

	  if (idx_found == irgb)	/* success, reorder and write a line */
	    {
	      dest = reorder;

	      if (scanner->params.depth > 8)
		{
		  for (j = scanner->params.pixels_per_line; j > 0; j--)
		    for (i = 0; i < irgb_out; i++)
		      {
			*dest++ = *src[i]++;
			*dest++ = *src[i]++;
		      }
		}
	      else
		{
		  for (j = scanner->params.pixels_per_line; j > 0; j--)
		    for (i = 0; i < irgb_out; i++)
		      *dest++ = *src[i]++;
		}
	      fwrite (reorder, 1, scanner->params.bytes_per_line, fp);
	      for (i = 0; i < irgb; i++)
		src[i] = NULL;
	      idx_found = 0;
	      write_lines--;
	    }
	}
      else			/* one color plane cases, write through */
	{
	  fwrite (idx_ptr[1], 1, size, fp);
	  idx_ptr[1] = (char *) wrt_ptr;
	  write_lines -= lines_todo;
	  if (write_lines > 0)
	    request_data = 1;
	}

      if (request_data)		/* read next data */
	{
	  if (read_lines)
	    {
	      lines_todo = chunk_lines;
	      if (lines_todo > read_lines)
		lines_todo = read_lines;
	      set_read_length (sread.cmd, lines_todo * irgb);
	      size = lines_todo * bytes_per_line;

	      do
		{
		  status =
		    pie_usb_scsi_wrapper (scanner->sfd, sread.cmd, sread.size,
					  wrt_ptr, &size);
		}
	      while (status);

	      DBG_DUMP (DBG_dump, wrt_ptr, 32);

	      read_lines -= lines_todo;
	      wrt_ptr += chunk_size;
	      if (wrt_ptr >= end_ptr)
		wrt_ptr = buffer;
	      for (i = 0; i < irgb; i++)
		idx_buf[i]++;
	      request_data = 0;
	    }
	  else if (write_lines)
	    {
	      DBG (DBG_error,
		   "pie_usb_reader_process_indexed: deskew failed for %d lines\n",
		   write_lines);
	      write_lines = 0;
	    }
	}
    }
  free (buffer);
  free (reorder);
  return 0;
}


/* ------------------------- PIE_USB_READER_PROCESS --------------------------- */

static int
pie_usb_reader_process (Pie_Scanner * scanner, FILE * fp)
{
  int status;
  int lines, lines_todo, chunk_lines;
  unsigned char *buffer, *reorder;
  size_t size, chunk_size;

  DBG (DBG_read, "pie_usb_reader_process reading %d lines of %d bytes/line\n",
       scanner->params.lines, scanner->params.bytes_per_line);

  chunk_lines = (scanner->params.lines + 7) / 8;
  chunk_size = chunk_lines * scanner->params.bytes_per_line;
  if (chunk_size > BUFFER_MAXSIZE)
    {
      chunk_lines = BUFFER_MAXSIZE / scanner->params.bytes_per_line;
      chunk_size = chunk_lines * scanner->params.bytes_per_line;
    }
  buffer = malloc (chunk_size);
  reorder = malloc (chunk_size);
  if (!buffer || !reorder)
    {
      free (buffer);
      free (reorder);
      return SANE_STATUS_NO_MEM;
    }

  lines = scanner->params.lines;

  while (lines > 0)
    {
      lines_todo = chunk_lines;
      if (lines_todo > lines)
	lines_todo = lines;
      set_read_length (sread.cmd, lines_todo);
      size = lines_todo * scanner->params.bytes_per_line;

      do
	{
	  status =
	    pie_usb_scsi_wrapper (scanner->sfd, sread.cmd, sread.size, buffer,
				  &size);
	}
      while (status);

      DBG_DUMP (DBG_dump, buffer, 64);

      if (scanner->colormode == RGB)
	{
	  int offset = scanner->bytes_per_line;
	  unsigned char *dest = reorder;
	  unsigned char *src = buffer;
	  int i, j;

	  if (scanner->params.depth > 8)
	    {
	      for (j = lines_todo; j > 0; j--)
		{
		  for (i = scanner->params.pixels_per_line; i > 0; i--)
		    {
		      *dest++ = *src;
		      *dest++ = *(src + 1);
		      *dest++ = *(src + offset);
		      *dest++ = *(src + offset + 1);
		      *dest++ = *(src + 2 * offset);
		      *dest++ = *(src + 2 * offset + 1);
		      src++;
		      src++;
		    }
		  src += offset * 2;
		}
	    }
	  else
	    {
	      for (j = lines_todo; j > 0; j--)
		{
		  for (i = scanner->params.pixels_per_line; i > 0; i--)
		    {
		      *dest++ = *src;
		      *dest++ = *(src + offset);
		      *dest++ = *(src + 2 * offset);
		      src++;
		    }
		  src += offset * 2;
		}
	    }
	  fwrite (reorder, 1, size, fp);
	}
      else
	{
	  fwrite (buffer, 1, size, fp);
	}
      lines -= lines_todo;
    }

  free (buffer);
  free (reorder);

  return 0;
}

